<?php

namespace ShopManagerPro\Products\Fields;

use ShopManagerPro\ACF\ACFRepository;
use ShopManagerPro\Products\Fields\DTO\FieldIdentifier;
use ShopManagerPro\Products\Fields\DTO\ProductFieldGroupType;

class FieldTypeMapper {
	private static array $fieldTypeMappings = ['text' => ['operators' => ['equals', 'contains', 'starts_with', 'ends_with', 'empty'], 'filterMethod' => 'text'], 'textarea' => ['operators' => ['equals', 'contains', 'starts_with', 'ends_with', 'empty'], 'filterMethod' => 'text'], 'integer' => ['operators' => ['equals', 'greater_than', 'greater_than_or_equal', 'less_than', 'less_than_or_equal', 'empty'], 'filterMethod' => 'numeric'], 'positive_integer' => ['operators' => ['equals', 'greater_than', 'greater_than_or_equal', 'less_than', 'less_than_or_equal', 'empty'], 'filterMethod' => 'numeric'], 'float' => ['operators' => ['equals', 'greater_than', 'greater_than_or_equal', 'less_than', 'less_than_or_equal', 'empty'], 'filterMethod' => 'numeric'], 'currency' => ['operators' => ['equals', 'greater_than', 'greater_than_or_equal', 'less_than', 'less_than_or_equal', 'empty'], 'filterMethod' => 'numeric'], 'boolean' => ['operators' => ['equals'], 'filterMethod' => 'boolean'], 'select' => ['operators' => ['equals', 'in', 'contains', 'starts_with', 'ends_with', 'empty'], 'filterMethod' => 'select'], 'multiselect' => ['operators' => ['equals', 'in', 'contains', 'starts_with', 'ends_with', 'empty'], 'filterMethod' => 'multiselect'], 'datetime' => ['operators' => ['equals', 'greater_than', 'less_than', 'empty'], 'filterMethod' => 'date']];

	public static function resolveFieldType(FieldIdentifier $field) {
		switch ($field->type) {
			case ProductFieldGroupType::ACF: return self::resolveACFFieldType($field->key);
			case ProductFieldGroupType::META: if (defined('SHOP_MANAGER_PRO_COMMERCIAL') && SHOP_MANAGER_PRO_COMMERCIAL) {
				$customFields = \ShopManagerPro\Premium\CustomFields\CustomFieldDiscoveryService::getActiveCustomFields();
				if (isset($customFields->byKey[$field->key])) {
					$dataType = $customFields->byKey[$field->key]->getDataType()->value;

					return match ($dataType) {
						'float' => 'float', 'integer' => 'integer', 'boolean' => 'boolean', 'date' => 'datetime', default => 'text',
					};
				}
			}

return 'text';
			case ProductFieldGroupType::DEFAULT_TAXONOMY: case ProductFieldGroupType::PRODUCT_ATTRIBUTE: return 'multiselect';
			case ProductFieldGroupType::BASIC: return self::resolveBasicFieldType($field->key);
			default: return 'text';
		}
	}

	private static function resolveACFFieldType(string $fieldName) {
		$acfCollection = ACFRepository::getACFFields();
		if (isset($acfCollection->acfFieldsByName[$fieldName])) {
			$field = $acfCollection->acfFieldsByName[$fieldName];
			$acfType = $field->type;
			if ($acfType === 'select' || $acfType === 'checkbox') {
				return $field->multiple ? 'multiselect' : 'select';
			} if ($acfType === 'true_false') {
				return 'boolean';
			} if ($acfType === 'radio') {
				return 'select';
			} if ($acfType === 'number') {
				return 'integer';
			} if ($acfType === 'textarea' || $acfType === 'wysiwyg') {
				return 'textarea';
			} if ($acfType === 'date_picker' || $acfType === 'date_time_picker') {
				return 'datetime';
			}
		}

return 'text';
	}

	private static function resolveBasicFieldType(string $fieldName) {
		return match ($fieldName) {
			'regularPrice', 'salePrice' => 'currency', 'stock' => 'positive_integer', 'stockStatus', 'postStatus' => 'select', 'name', 'sku' => 'text', 'galleryImageIds' => 'multiselect', default => 'text',
		};
	}

	public static function getFieldConfig(FieldIdentifier $field) {
		$fieldType = self::resolveFieldType($field);

		return self::$fieldTypeMappings[$fieldType] ?? self::$fieldTypeMappings['text'];
	}

	public static function isValidOperator(FieldIdentifier $field, string $operator) {
		$config = self::getFieldConfig($field);

		return in_array($operator, $config['operators'], true);
	}

	public static function getFilterMethod(FieldIdentifier $field) {
		$config = self::getFieldConfig($field);

		return $config['filterMethod'];
	}
}
