# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Directory Overview

The `includes/` directory contains the core PHP source code for the Shop Manager Pro WordPress plugin. This is where all the business logic, REST API endpoints, and service classes are implemented.

## Directory Structure

### Core Feature Modules
- **ACF/** - Advanced Custom Fields integration (reads ACF field groups and fields)
- **Filters/** - Product filtering logic and REST API
- **Images/** - Product image management and REST API
- **Jobs/** - Background job processing using WordPress Action Scheduler
- **Products/** - Core product management functionality
- **Taxonomies/** - WooCommerce taxonomy handling
- **TaxClasses/** - Tax class management
- **System/** - System information API

### Premium Features (Commercial Version Only)
- **Premium/** - Contains all premium features
  - **CustomFields/** - Custom field discovery and management
  - **Export/** - CSV export functionality
  - **License/** - License validation system
  - **Filters/** - Advanced filtering capabilities

### Shared Components
- **Shared/** - Common utilities and services used across modules
  - **Utils/** - Helper classes (Constants, JSON, Logger, SQL, Time)
  - Core services (CapabilityService, DatabaseManagerService, InitService)

## Key Architecture Patterns

### 1. API Class Pattern
Each feature module has an API class (e.g., `ProductAPI.php`) that:
```php
class FeatureAPI {
    // OpenAPI-style schema definitions
    public static $SCHEMA_*;
    
    // Initialization
    public static function init() {
        self::registerRoutes();
        self::initSchemas();
    }
    
    // REST route registration
    public static function registerRoutes() {
        register_rest_route(Constants::API_NAMESPACE, '/endpoint', [...]);
    }
}
```

### 2. Repository Pattern
Data access is handled through Repository classes:
```php
class FeatureRepository {
    // Returns DTO instances
    public static function findById(int $id): ?FeatureDTO
    // Uses wpdb for database queries
    // Utilizes SQL helper for query building
}
```

### 3. DTO Structure
- **DTO/** - Data Transfer Objects for type safety
- **DTO/Generated/** - Auto-generated from OpenAPI schemas (DO NOT EDIT)
- DTOs use typed properties and are instantiated from database results

### 4. Service Classes
Business logic encapsulated in services:
- `*Service.php` - Contains complex operations
- Coordinates between repositories and APIs
- Handles transactions and error logging

## Code Standards

### Namespace Convention
All classes use the `ShopManagerPro\{Module}\{Submodule}` namespace pattern.

### Static Methods
Most classes use static methods due to WordPress's procedural nature and hook system.

### Error Handling
```php
// Always use Logger for errors
\ShopManagerPro\Shared\Utils\Logger::error('Message', ['context' => $data]);

// Return WP_Error for REST API errors
return new \WP_Error('error_code', 'Error message', ['status' => 400]);
```

### Permission Checks
All REST endpoints use CapabilityService:
```php
'permission_callback' => [\ShopManagerPro\Shared\CapabilityService::class, 'requireReadCapability']
// or
'permission_callback' => [\ShopManagerPro\Shared\CapabilityService::class, 'requireEditCapability']
```

## Important Implementation Details

### Products Module
- **ProductUpdateService** - Handles complex product updates with validation
- **Fields/** - Manages different field types (Basic, ACF, Custom Meta, Taxonomies, Attributes)
- **Groups/** - Handles bulk product operations

### Jobs System
- Uses WordPress Action Scheduler for background processing
- Processes in batches (default: 20 items)
- Maximum execution time: 15 seconds per batch
- Job statuses: pending, running, completed, failed, stopped

### Database Operations
- Use `global $wpdb` for queries
- Utilize `\ShopManagerPro\Shared\Utils\SQL` for building WHERE clauses
- Tables use `{$wpdb->prefix}shopmanagerpro_*` naming

### REST API Conventions
- Namespace: `/shopmanagerpro/`
- GET for reading data
- PUT for updates (with ID in URL)
- POST for creation
- DELETE for removal

## Development Notes

### Adding New Features
1. Create module directory under `includes/`
2. Implement API class with schemas
3. Create Repository for data access
4. Add DTOs (generate if needed)
5. Register in `Shared/InitService.php`

### Modifying Schemas
1. Update schema in API class
2. Regenerate DTOs if using Generated classes
3. Run `npm run openapi-smp` in frontend to update TypeScript client

### Testing Considerations
- Unit tests go in `plugin-tests/unit/`
- Integration tests in `plugin-tests/integration/`
- Use fixtures for consistent test data